
/*
 * Simulazione del modello di calcolo Unlimited Register Machine
 * (N. Cutland, "Computability", Cambridge University Press)
 *
 * Tipica sequenza di utilizzo:
 *
 *   M = new URM(P);
 *   M.input(1,x1);
 *     ...
 *   M.input(n,xn);
 *   M.run();
 *   y = M.output(1);
 *
 * dove P e' un programma per calcolare la funzione n-aria f
 * i cui valori sono definiti da  y = f(x1,...,xn).
 *
 * Osservazioni:
 * Dopo che la macchina e' stata creata con programma P,
 * si assume che P non venga piu' modificato.
 */


public class URM {


  // Registri: R[1], ..., R[regs]
  
  private int[] R;
  private final int regs;
  
  // Programma: indici istruzioni 1, ..., s
  
  private final Program program;
  private final int s;
  private int p;  // Indice istruzione corrente
  
  
  public URM( Program program ) {
  
    this.program = program;
    
    s = program.size();
    regs = program.upperRegisterIndex();
    R = new int[regs+1];  // R[0] inutilizzato
  }
  
  
  // Operazioni elementari
  // Si assume che i parametri siano consistenti
  
  public void zero( int n ) {                 // 1 <= n <= regs
  
    R[n] = 0;
  }
  
  public void succ( int n ) {                 // 1 <= n <= regs
  
    R[n] = R[n] + 1;
  }
  
  public void tran( int m, int n ) {         // 1 <= m, n <= regs
  
    R[n] = R[m];
  }
  
  public void jump( int m, int n, int q ) {  // 1 <= m, n <= regs, q >= 1
  
    if ( R[m] == R[n] ) {
      p = q;
    }
  }
  
  
  // Scrittura e lettura dei dati contenuti nei registri
  // Si assume che i parametri siano consistenti
  
  public void input( int i, int x ) {        // 1 <= i <= regs, x >= 0
  
    if ( (i >= 1) && (i <= regs) ) {
      R[i] = x;
    }
  }
  
  public int output( int i ) {               // 1 <= i <= regs
  
    if ( (i >= 1) && (i <= regs) ) {
      return R[i];
    } else {
      return 0;
    }
  }
  
  
  // Ciclo di esecuzione del programma
  
  public void run() {
  
    p = 1;
    while ( p <= s ) {
      Instruction I = program.fetch(p);
      p = p + 1;
      I.exec( this );                         // exec polimorfo
    }
  }

}  // class URM
